(function($) {
    'use strict';
    
    class SongVoterWidget {
        constructor(container) {
            this.container = container;
            this.mediaId = container.data('media-id');
            this.audio = container.find('.svw-audio-element')[0];
            this.playButton = container.find('.svw-play-button');
            this.progressContainer = container.find('.svw-progress-container');
            this.progressBar = container.find('.svw-progress-bar');
            this.currentTimeEl = container.find('.svw-current-time');
            this.durationEl = container.find('.svw-duration');
            this.voteButtons = container.find('.svw-vote-button');
            
            this.init();
        }
        
        init() {
            this.bindEvents();
            this.loadAudioMetadata();
        }
        
        bindEvents() {
            // Play/Pause
            this.playButton.on('click', () => this.togglePlay());
            
            // Progress bar click
            this.progressContainer.on('click', (e) => this.seek(e));
            
            // Audio events
            $(this.audio).on('timeupdate', () => this.updateProgress());
            $(this.audio).on('loadedmetadata', () => this.updateDuration());
            $(this.audio).on('ended', () => this.onEnded());
            
            // Voting
            this.voteButtons.on('click', (e) => this.handleVote(e));
        }
        
        loadAudioMetadata() {
            if (this.audio) {
                this.audio.load();
            }
        }
        
        togglePlay() {
            if (this.audio.paused) {
                this.audio.play();
                this.playButton.addClass('playing');
            } else {
                this.audio.pause();
                this.playButton.removeClass('playing');
            }
        }
        
        seek(e) {
            const percent = e.offsetX / this.progressContainer.width();
            this.audio.currentTime = percent * this.audio.duration;
        }
        
        updateProgress() {
            if (this.audio.duration) {
                const percent = (this.audio.currentTime / this.audio.duration) * 100;
                this.progressBar.css('width', percent + '%');
                this.currentTimeEl.text(this.formatTime(this.audio.currentTime));
            }
        }
        
        updateDuration() {
            if (this.audio.duration && !isNaN(this.audio.duration)) {
                this.durationEl.text(this.formatTime(this.audio.duration));
            }
        }
        
        onEnded() {
            this.playButton.removeClass('playing');
            this.progressBar.css('width', '0%');
        }
        
        formatTime(seconds) {
            if (isNaN(seconds)) return '0:00';
            
            const mins = Math.floor(seconds / 60);
            const secs = Math.floor(seconds % 60);
            return mins + ':' + (secs < 10 ? '0' : '') + secs;
        }
        
        handleVote(e) {
            e.preventDefault();
            
            const button = $(e.currentTarget);
            const voteType = button.data('vote-type');
            const wasVoted = button.hasClass('voted');
            
            // Prevent multiple clicks
            if (button.hasClass('voting')) {
                return;
            }
            
            // Determine action
            let action = voteType;
            if (wasVoted) {
                // Clicking the same button again removes the vote
                action = 'remove';
            }
            
            // Add loading state
            button.addClass('voting');
            
            // Send AJAX request
            $.ajax({
                url: svwAjax.ajaxurl,
                type: 'POST',
                data: {
                    action: 'svw_vote',
                    nonce: svwAjax.nonce,
                    media_id: this.mediaId,
                    vote_type: action
                },
                success: (response) => {
                    button.removeClass('voting');
                    
                    if (response.success) {
                        this.updateVoteDisplay(response.data, voteType, wasVoted);
                        
                        // Visual feedback
                        button.addClass('vote-success');
                        setTimeout(() => {
                            button.removeClass('vote-success');
                        }, 400);
                    } else {
                        console.error('Vote failed:', response.data.message);
                        this.showError(button);
                    }
                },
                error: (xhr, status, error) => {
                    button.removeClass('voting');
                    console.error('AJAX error:', error);
                    this.showError(button);
                }
            });
        }
        
        updateVoteDisplay(data, clickedVote, wasVoted) {
            // Update vote counts
            this.container.find('.svw-vote-up .svw-vote-count').text(data.thumbs_up);
            this.container.find('.svw-vote-down .svw-vote-count').text(data.thumbs_down);
            
            // Update button states
            this.voteButtons.removeClass('voted');
            
            if (wasVoted) {
                // Vote was removed, no button should be active
                return;
            }
            
            // If switching votes or new vote
            const otherVote = clickedVote === 'up' ? 'down' : 'up';
            this.container.find('.svw-vote-' + otherVote).removeClass('voted');
            this.container.find('.svw-vote-' + clickedVote).addClass('voted');
        }
        
        showError(button) {
            const originalBg = button.css('background');
            button.css('background', 'linear-gradient(145deg, #fecaca, #fee2e2)');
            
            setTimeout(() => {
                button.css('background', originalBg);
            }, 1000);
        }
    }
    
    // Initialize widgets when DOM is ready
    $(document).ready(function() {
        initializeWidgets();
    });
    
    // Also try initialization on window load as backup
    $(window).on('load', function() {
        initializeWidgets();
    });
    
    // Re-initialize when Elementor preview updates
    $(window).on('elementor/frontend/init', function() {
        elementorFrontend.hooks.addAction('frontend/element_ready/song_voter_widget.default', function($scope) {
            const container = $scope.find('.svw-player-container');
            if (container.length) {
                new SongVoterWidget(container);
            }
        });
    });
    
    function initializeWidgets() {
        $('.svw-player-container').each(function() {
            const $container = $(this);
            // Check if already initialized
            if (!$container.data('svw-initialized')) {
                new SongVoterWidget($container);
                $container.data('svw-initialized', true);
            }
        });
    }
    
})(jQuery);
