<?php
/**
 * AJAX Handler Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class SVW_Ajax_Handler {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('wp_ajax_svw_vote', array($this, 'handle_vote'));
        add_action('wp_ajax_nopriv_svw_vote', array($this, 'handle_vote'));
        add_action('wp_ajax_svw_get_votes', array($this, 'get_votes'));
        add_action('wp_ajax_nopriv_svw_get_votes', array($this, 'get_votes'));
    }
    
    /**
     * Handle vote submission
     */
    public function handle_vote() {
        // Verify nonce
        if (!check_ajax_referer('svw_vote_nonce', 'nonce', false)) {
            wp_send_json_error(array('message' => 'Invalid security token'));
            return;
        }
        
        // Get and validate input
        $media_id = isset($_POST['media_id']) ? intval($_POST['media_id']) : 0;
        $vote_type = isset($_POST['vote_type']) ? sanitize_text_field($_POST['vote_type']) : '';
        
        if (!$media_id || !in_array($vote_type, array('up', 'down', 'remove'))) {
            wp_send_json_error(array('message' => 'Invalid vote data'));
            return;
        }
        
        // Check rate limiting (simple: 1 vote per second per IP)
        $this->check_rate_limit();
        
        // Get previous vote from cookie
        $cookie_name = 'svw_vote_' . $media_id;
        $previous_vote = isset($_COOKIE[$cookie_name]) ? sanitize_text_field($_COOKIE[$cookie_name]) : null;
        
        $db = SVW_Database::get_instance();
        
        // Handle vote removal
        if ($vote_type === 'remove') {
            if ($previous_vote) {
                $result = $db->remove_vote($media_id, $previous_vote);
                setcookie($cookie_name, '', time() - 3600, COOKIEPATH, COOKIE_DOMAIN);
                wp_send_json_success($result);
            } else {
                wp_send_json_error(array('message' => 'No vote to remove'));
            }
            return;
        }
        
        // Update vote in database
        $result = $db->update_vote($media_id, $vote_type, $previous_vote);
        
        if ($result['success']) {
            // Set cookie for 1 year
            setcookie($cookie_name, $vote_type, time() + YEAR_IN_SECONDS, COOKIEPATH, COOKIE_DOMAIN);
            wp_send_json_success($result);
        } else {
            wp_send_json_error($result);
        }
    }
    
    /**
     * Get current vote counts
     */
    public function get_votes() {
        $media_id = isset($_GET['media_id']) ? intval($_GET['media_id']) : 0;
        
        if (!$media_id) {
            wp_send_json_error(array('message' => 'Invalid media ID'));
            return;
        }
        
        $db = SVW_Database::get_instance();
        $votes = $db->get_votes($media_id);
        
        // Get user's current vote from cookie
        $cookie_name = 'svw_vote_' . $media_id;
        $user_vote = isset($_COOKIE[$cookie_name]) ? sanitize_text_field($_COOKIE[$cookie_name]) : null;
        
        wp_send_json_success(array(
            'thumbs_up' => $votes['thumbs_up'],
            'thumbs_down' => $votes['thumbs_down'],
            'user_vote' => $user_vote
        ));
    }
    
    /**
     * Simple rate limiting
     */
    private function check_rate_limit() {
        $ip = $_SERVER['REMOTE_ADDR'];
        $transient_key = 'svw_rate_limit_' . md5($ip);
        
        if (get_transient($transient_key)) {
            wp_send_json_error(array('message' => 'Please wait before voting again'));
            exit;
        }
        
        // Set transient for 1 second
        set_transient($transient_key, true, 1);
    }
}
