<?php
/**
 * Database Handler Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class SVW_Database {
    
    private static $instance = null;
    private $table_name;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'song_votes';
    }
    
    /**
     * Create the votes table
     */
    public static function create_table() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'song_votes';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            media_id bigint(20) UNSIGNED NOT NULL,
            thumbs_up int(11) DEFAULT 0,
            thumbs_down int(11) DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY  (id),
            UNIQUE KEY media_id (media_id)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Get vote counts for a media item
     */
    public function get_votes($media_id) {
        global $wpdb;
        
        $result = $wpdb->get_row($wpdb->prepare(
            "SELECT thumbs_up, thumbs_down FROM {$this->table_name} WHERE media_id = %d",
            $media_id
        ));
        
        if ($result) {
            return array(
                'thumbs_up' => (int) $result->thumbs_up,
                'thumbs_down' => (int) $result->thumbs_down
            );
        }
        
        return array(
            'thumbs_up' => 0,
            'thumbs_down' => 0
        );
    }
    
    /**
     * Add or update a vote
     */
    public function update_vote($media_id, $vote_type, $previous_vote = null) {
        global $wpdb;
        
        // Get current votes
        $current_votes = $this->get_votes($media_id);
        
        $thumbs_up = $current_votes['thumbs_up'];
        $thumbs_down = $current_votes['thumbs_down'];
        
        // Remove previous vote if exists
        if ($previous_vote === 'up') {
            $thumbs_up = max(0, $thumbs_up - 1);
        } elseif ($previous_vote === 'down') {
            $thumbs_down = max(0, $thumbs_down - 1);
        }
        
        // Add new vote
        if ($vote_type === 'up') {
            $thumbs_up++;
        } elseif ($vote_type === 'down') {
            $thumbs_down++;
        }
        
        // Insert or update
        $result = $wpdb->replace(
            $this->table_name,
            array(
                'media_id' => $media_id,
                'thumbs_up' => $thumbs_up,
                'thumbs_down' => $thumbs_down
            ),
            array('%d', '%d', '%d')
        );
        
        if ($result !== false) {
            return array(
                'success' => true,
                'thumbs_up' => $thumbs_up,
                'thumbs_down' => $thumbs_down
            );
        }
        
        return array(
            'success' => false,
            'message' => 'Failed to update vote'
        );
    }
    
    /**
     * Remove a vote
     */
    public function remove_vote($media_id, $vote_type) {
        global $wpdb;
        
        $current_votes = $this->get_votes($media_id);
        
        $thumbs_up = $current_votes['thumbs_up'];
        $thumbs_down = $current_votes['thumbs_down'];
        
        if ($vote_type === 'up') {
            $thumbs_up = max(0, $thumbs_up - 1);
        } elseif ($vote_type === 'down') {
            $thumbs_down = max(0, $thumbs_down - 1);
        }
        
        $wpdb->replace(
            $this->table_name,
            array(
                'media_id' => $media_id,
                'thumbs_up' => $thumbs_up,
                'thumbs_down' => $thumbs_down
            ),
            array('%d', '%d', '%d')
        );
        
        return array(
            'success' => true,
            'thumbs_up' => $thumbs_up,
            'thumbs_down' => $thumbs_down
        );
    }
}
