<?php
/**
 * Elementor Widget Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class SVW_Elementor_Widget extends \Elementor\Widget_Base {
    
    public function get_name() {
        return 'song_voter_widget';
    }
    
    public function get_title() {
        return __('Song Voter', 'song-voter-widget');
    }
    
    public function get_icon() {
        return 'eicon-play';
    }
    
    public function get_categories() {
        return array('general');
    }
    
    public function get_keywords() {
        return array('song', 'audio', 'mp3', 'vote', 'rating', 'player');
    }
    
    protected function register_controls() {
        
        // Content Section
        $this->start_controls_section(
            'content_section',
            array(
                'label' => __('Content', 'song-voter-widget'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            )
        );
        
        $this->add_control(
            'audio_file',
            array(
                'label' => __('Choose Audio File', 'song-voter-widget'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'media_types' => array('audio'),
                'default' => array(
                    'url' => '',
                ),
            )
        );
        
        $this->add_control(
            'show_filename',
            array(
                'label' => __('Show Filename', 'song-voter-widget'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => __('Show', 'song-voter-widget'),
                'label_off' => __('Hide', 'song-voter-widget'),
                'return_value' => 'yes',
                'default' => 'yes',
            )
        );
        
        $this->end_controls_section();
        
        // Style Section
        $this->start_controls_section(
            'style_section',
            array(
                'label' => __('Style', 'song-voter-widget'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            )
        );
        
        $this->add_control(
            'player_width',
            array(
                'label' => __('Player Width', 'song-voter-widget'),
                'type' => \Elementor\Controls_Manager::SLIDER,
                'size_units' => array('%', 'px'),
                'range' => array(
                    '%' => array(
                        'min' => 50,
                        'max' => 100,
                    ),
                    'px' => array(
                        'min' => 300,
                        'max' => 1200,
                    ),
                ),
                'default' => array(
                    'unit' => '%',
                    'size' => 100,
                ),
                'selectors' => array(
                    '{{WRAPPER}} .svw-player-container' => 'max-width: {{SIZE}}{{UNIT}};',
                ),
            )
        );
        
        $this->add_control(
            'accent_color',
            array(
                'label' => __('Accent Color', 'song-voter-widget'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'default' => '#4a90e2',
                'selectors' => array(
                    '{{WRAPPER}} .svw-play-button:hover' => 'background: {{VALUE}};',
                    '{{WRAPPER}} .svw-vote-button.voted' => 'color: {{VALUE}};',
                    '{{WRAPPER}} .svw-progress-bar' => 'background: {{VALUE}};',
                ),
            )
        );
        
        $this->end_controls_section();
    }
    
    protected function render() {
        $settings = $this->get_settings_for_display();
        
        if (empty($settings['audio_file']['url'])) {
            echo '<p>' . __('Please select an audio file.', 'song-voter-widget') . '</p>';
            return;
        }
        
        $audio_url = $settings['audio_file']['url'];
        $media_id = $settings['audio_file']['id'];
        $show_filename = $settings['show_filename'] === 'yes';
        
        // Get filename
        $filename = basename($audio_url);
        if ($media_id) {
            $attachment_title = get_the_title($media_id);
            if ($attachment_title) {
                $filename = $attachment_title;
            }
        }
        
        // Get current votes
        $db = SVW_Database::get_instance();
        $votes = $db->get_votes($media_id);
        
        // Get user's current vote from cookie
        $cookie_name = 'svw_vote_' . $media_id;
        $user_vote = isset($_COOKIE[$cookie_name]) ? sanitize_text_field($_COOKIE[$cookie_name]) : '';
        
        ?>
        <div class="svw-player-container" data-media-id="<?php echo esc_attr($media_id); ?>">
            <div class="svw-player-main">
                <?php if ($show_filename) : ?>
                    <div class="svw-song-title">
                        <span class="svw-title-text"><?php echo esc_html($filename); ?></span>
                    </div>
                <?php endif; ?>
                
                <div class="svw-player-controls-row">
                    <button class="svw-play-button" aria-label="Play/Pause">
                        <svg class="svw-play-icon" viewBox="0 0 24 24" width="24" height="24">
                            <path fill="currentColor" d="M8 5v14l11-7z"/>
                        </svg>
                        <svg class="svw-pause-icon" viewBox="0 0 24 24" width="24" height="24" style="display: none;">
                            <path fill="currentColor" d="M6 4h4v16H6V4zm8 0h4v16h-4V4z"/>
                        </svg>
                    </button>
                    
                    <div class="svw-player-controls">
                        <div class="svw-progress-container">
                            <div class="svw-progress-bar"></div>
                        </div>
                        <div class="svw-time-info">
                            <span class="svw-current-time">0:00</span>
                            <span class="svw-duration">0:00</span>
                        </div>
                    </div>
                    
                    <div class="svw-voting-controls">
                        <button class="svw-vote-button svw-vote-up <?php echo $user_vote === 'up' ? 'voted' : ''; ?>" 
                                data-vote-type="up" 
                                aria-label="Thumbs Up">
                            <svg viewBox="0 0 24 24" width="20" height="20">
                                <path fill="currentColor" d="M1 21h4V9H1v12zm22-11c0-1.1-.9-2-2-2h-6.31l.95-4.57.03-.32c0-.41-.17-.79-.44-1.06L14.17 1 7.59 7.59C7.22 7.95 7 8.45 7 9v10c0 1.1.9 2 2 2h9c.83 0 1.54-.5 1.84-1.22l3.02-7.05c.09-.23.14-.47.14-.73v-2z"/>
                            </svg>
                            <span class="svw-vote-count"><?php echo esc_html($votes['thumbs_up']); ?></span>
                        </button>
                        
                        <button class="svw-vote-button svw-vote-down <?php echo $user_vote === 'down' ? 'voted' : ''; ?>" 
                                data-vote-type="down" 
                                aria-label="Thumbs Down">
                            <svg viewBox="0 0 24 24" width="20" height="20">
                                <path fill="currentColor" d="M15 3H6c-.83 0-1.54.5-1.84 1.22l-3.02 7.05c-.09.23-.14.47-.14.73v2c0 1.1.9 2 2 2h6.31l-.95 4.57-.03.32c0 .41.17.79.44 1.06L9.83 23l6.59-6.59c.36-.36.58-.86.58-1.41V5c0-1.1-.9-2-2-2zm4 0v12h4V3h-4z"/>
                            </svg>
                            <span class="svw-vote-count"><?php echo esc_html($votes['thumbs_down']); ?></span>
                        </button>
                    </div>
                </div>
            </div>
                            <path fill="currentColor" d="M1 21h4V9H1v12zm22-11c0-1.1-.9-2-2-2h-6.31l.95-4.57.03-.32c0-.41-.17-.79-.44-1.06L14.17 1 7.59 7.59C7.22 7.95 7 8.45 7 9v10c0 1.1.9 2 2 2h9c.83 0 1.54-.5 1.84-1.22l3.02-7.05c.09-.23.14-.47.14-.73v-2z"/>
                        </svg>
                        <span class="svw-vote-count"><?php echo esc_html($votes['thumbs_up']); ?></span>
                    </button>
                    
                    <button class="svw-vote-button svw-vote-down <?php echo $user_vote === 'down' ? 'voted' : ''; ?>" 
                            data-vote-type="down" 
                            aria-label="Thumbs Down">
                        <svg viewBox="0 0 24 24" width="20" height="20">
                            <path fill="currentColor" d="M15 3H6c-.83 0-1.54.5-1.84 1.22l-3.02 7.05c-.09.23-.14.47-.14.73v2c0 1.1.9 2 2 2h6.31l-.95 4.57-.03.32c0 .41.17.79.44 1.06L9.83 23l6.59-6.59c.36-.36.58-.86.58-1.41V5c0-1.1-.9-2-2-2zm4 0v12h4V3h-4z"/>
                        </svg>
                        <span class="svw-vote-count"><?php echo esc_html($votes['thumbs_down']); ?></span>
                    </button>
                </div>
            </div>
            
            <audio class="svw-audio-element" preload="metadata">
                <source src="<?php echo esc_url($audio_url); ?>" type="audio/mpeg">
                Your browser does not support the audio element.
            </audio>
        </div>
        <?php
    }
}
