<?php
/**
 * Gutenberg Block Handler Class
 */

if (!defined('ABSPATH')) {
    exit;
}

class SVW_Gutenberg_Block {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('init', array($this, 'register_block'));
        add_action('enqueue_block_editor_assets', array($this, 'enqueue_block_editor_assets'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_frontend_assets'));
    }
    
    public function enqueue_frontend_assets() {
        // Always enqueue on frontend for maximum compatibility
        wp_enqueue_style(
            'song-voter-widget',
            SVW_PLUGIN_URL . 'assets/css/style.css',
            array(),
            SVW_VERSION
        );
        
        wp_enqueue_script(
            'song-voter-widget',
            SVW_PLUGIN_URL . 'assets/js/script.js',
            array('jquery'),
            SVW_VERSION,
            true
        );
        
        wp_localize_script('song-voter-widget', 'svwAjax', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('svw_vote_nonce')
        ));
    }
    
    public function register_block() {
        if (!function_exists('register_block_type')) {
            return;
        }
        
        register_block_type('song-voter-widget/audio-voter', array(
            'editor_script' => 'svw-block-editor',
            'render_callback' => array($this, 'render_block'),
            'attributes' => array(
                'mediaId' => array(
                    'type' => 'number',
                    'default' => 0,
                ),
                'showFilename' => array(
                    'type' => 'boolean',
                    'default' => true,
                ),
                'width' => array(
                    'type' => 'string',
                    'default' => '100%',
                ),
            ),
        ));
    }
    
    public function enqueue_block_editor_assets() {
        wp_enqueue_script(
            'svw-block-editor',
            SVW_PLUGIN_URL . 'assets/js/block.js',
            array('wp-blocks', 'wp-element', 'wp-editor', 'wp-components', 'wp-i18n'),
            SVW_VERSION,
            true
        );
        
        wp_enqueue_style(
            'svw-block-editor',
            SVW_PLUGIN_URL . 'assets/css/block-editor.css',
            array('wp-edit-blocks'),
            SVW_VERSION
        );
    }
    
    public function render_block($attributes) {
        $media_id = isset($attributes['mediaId']) ? intval($attributes['mediaId']) : 0;
        $show_filename = isset($attributes['showFilename']) ? $attributes['showFilename'] : true;
        $width = isset($attributes['width']) ? sanitize_text_field($attributes['width']) : '100%';
        
        if (!$media_id) {
            return '<p>' . __('Please select an audio file.', 'song-voter-widget') . '</p>';
        }
        
        $audio_url = wp_get_attachment_url($media_id);
        
        if (!$audio_url) {
            return '<p>' . __('Audio file not found.', 'song-voter-widget') . '</p>';
        }
        
        // Get filename
        $filename = get_the_title($media_id);
        if (!$filename) {
            $filename = basename($audio_url);
        }
        
        // Get current votes
        $db = SVW_Database::get_instance();
        $votes = $db->get_votes($media_id);
        
        // Get user's current vote from cookie
        $cookie_name = 'svw_vote_' . $media_id;
        $user_vote = isset($_COOKIE[$cookie_name]) ? sanitize_text_field($_COOKIE[$cookie_name]) : '';
        
        ob_start();
        ?>
        <div class="svw-player-container" data-media-id="<?php echo esc_attr($media_id); ?>" style="max-width: <?php echo esc_attr($width); ?>;">
            <div class="svw-player-main">
                <?php if ($show_filename) : ?>
                    <div class="svw-song-title">
                        <span class="svw-title-text"><?php echo esc_html($filename); ?></span>
                    </div>
                <?php endif; ?>
                
                <div class="svw-player-controls-row">
                    <button class="svw-play-button" aria-label="Play/Pause">
                        <svg class="svw-play-icon" viewBox="0 0 24 24" width="24" height="24">
                            <path fill="currentColor" d="M8 5v14l11-7z"/>
                        </svg>
                        <svg class="svw-pause-icon" viewBox="0 0 24 24" width="24" height="24" style="display: none;">
                            <path fill="currentColor" d="M6 4h4v16H6V4zm8 0h4v16h-4V4z"/>
                        </svg>
                    </button>
                    
                    <div class="svw-player-controls">
                        <div class="svw-progress-container">
                            <div class="svw-progress-bar"></div>
                        </div>
                        <div class="svw-time-info">
                            <span class="svw-current-time">0:00</span>
                            <span class="svw-duration">0:00</span>
                        </div>
                    </div>
                    
                    <div class="svw-voting-controls">
                        <button class="svw-vote-button svw-vote-up <?php echo $user_vote === 'up' ? 'voted' : ''; ?>" 
                                data-vote-type="up" 
                                aria-label="Thumbs Up">
                            <svg viewBox="0 0 24 24" width="20" height="20">
                                <path fill="currentColor" d="M1 21h4V9H1v12zm22-11c0-1.1-.9-2-2-2h-6.31l.95-4.57.03-.32c0-.41-.17-.79-.44-1.06L14.17 1 7.59 7.59C7.22 7.95 7 8.45 7 9v10c0 1.1.9 2 2 2h9c.83 0 1.54-.5 1.84-1.22l3.02-7.05c.09-.23.14-.47.14-.73v-2z"/>
                            </svg>
                            <span class="svw-vote-count"><?php echo esc_html($votes['thumbs_up']); ?></span>
                        </button>
                        
                        <button class="svw-vote-button svw-vote-down <?php echo $user_vote === 'down' ? 'voted' : ''; ?>" 
                                data-vote-type="down" 
                                aria-label="Thumbs Down">
                            <svg viewBox="0 0 24 24" width="20" height="20">
                                <path fill="currentColor" d="M15 3H6c-.83 0-1.54.5-1.84 1.22l-3.02 7.05c-.09.23-.14.47-.14.73v2c0 1.1.9 2 2 2h6.31l-.95 4.57-.03.32c0 .41.17.79.44 1.06L9.83 23l6.59-6.59c.36-.36.58-.86.58-1.41V5c0-1.1-.9-2-2-2zm4 0v12h4V3h-4z"/>
                            </svg>
                            <span class="svw-vote-count"><?php echo esc_html($votes['thumbs_down']); ?></span>
                        </button>
                    </div>
                </div>
            </div>
            
            <audio class="svw-audio-element" preload="metadata">
                <source src="<?php echo esc_url($audio_url); ?>" type="audio/mpeg">
                Your browser does not support the audio element.
            </audio>
        </div>
        <?php
        return ob_get_clean();
    }
}
