<?php
/**
 * Plugin Name: Song Voter Widget
 * Plugin URI: https://kourentzes.com
 * Description: Allows people to vote on mp3 files
 * Version: 1.2.0
 * Author: konstantinos@kourentzes.com
 * Author URI: https://kourentzes.com
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: song-voter-widget
 * Requires at least: 5.0
 * Requires PHP: 7.2
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Define plugin constants
define('SVW_VERSION', '1.2.0');
define('SVW_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SVW_PLUGIN_URL', plugin_dir_url(__FILE__));

// Include required files
require_once SVW_PLUGIN_DIR . 'includes/class-database.php';
require_once SVW_PLUGIN_DIR . 'includes/class-ajax-handler.php';
require_once SVW_PLUGIN_DIR . 'includes/class-shortcode.php';
require_once SVW_PLUGIN_DIR . 'includes/class-gutenberg-block.php';

/**
 * Main Plugin Class
 */
class Song_Voter_Widget {
    
    private static $instance = null;
    
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    public function init() {
        // Initialize database
        SVW_Database::get_instance();
        
        // Initialize AJAX handler
        SVW_Ajax_Handler::get_instance();
        
        // Initialize shortcode (works everywhere)
        SVW_Shortcode::get_instance();
        
        // Initialize Gutenberg block (works everywhere)
        SVW_Gutenberg_Block::get_instance();
        
        // Check if Elementor is installed and activated
        if (did_action('elementor/loaded')) {
            require_once SVW_PLUGIN_DIR . 'includes/class-elementor-widget.php';
            
            // Initialize Elementor widget
            add_action('elementor/widgets/register', array($this, 'register_elementor_widgets'));
            add_action('elementor/frontend/after_enqueue_styles', array($this, 'enqueue_styles'));
            add_action('elementor/frontend/after_register_scripts', array($this, 'enqueue_scripts'));
        } else {
            add_action('admin_notices', array($this, 'elementor_missing_notice'));
        }
    }
    
    public function elementor_missing_notice() {
        if (isset($_GET['activate'])) {
            unset($_GET['activate']);
        }
        
        $message = sprintf(
            esc_html__('"%1$s" works with or without Elementor! You can use shortcodes [song_voter id="123"] or Gutenberg blocks. Install "%2$s" for the Elementor widget.', 'song-voter-widget'),
            '<strong>' . esc_html__('Song Voter Widget', 'song-voter-widget') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'song-voter-widget') . '</strong>'
        );
        
        printf('<div class="notice notice-info is-dismissible"><p>%1$s</p></div>', $message);
    }
    
    public function register_elementor_widgets($widgets_manager) {
        $widgets_manager->register(new \SVW_Elementor_Widget());
    }
    
    public function enqueue_styles() {
        wp_enqueue_style(
            'song-voter-widget',
            SVW_PLUGIN_URL . 'assets/css/style.css',
            array(),
            SVW_VERSION
        );
    }
    
    public function enqueue_scripts() {
        wp_enqueue_script(
            'song-voter-widget',
            SVW_PLUGIN_URL . 'assets/js/script.js',
            array('jquery'),
            SVW_VERSION,
            true
        );
        
        wp_localize_script('song-voter-widget', 'svwAjax', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('svw_vote_nonce')
        ));
    }
    
    public function enqueue_global_assets() {
        wp_register_style(
            'song-voter-widget',
            SVW_PLUGIN_URL . 'assets/css/style.css',
            array(),
            SVW_VERSION
        );
        
        wp_register_script(
            'song-voter-widget',
            SVW_PLUGIN_URL . 'assets/js/script.js',
            array('jquery'),
            SVW_VERSION,
            true
        );
        
        wp_localize_script('song-voter-widget', 'svwAjax', array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('svw_vote_nonce')
        ));
    }
    
    public function activate() {
        SVW_Database::create_table();
        flush_rewrite_rules();
    }
    
    public function deactivate() {
        flush_rewrite_rules();
    }
}

// Initialize the plugin
Song_Voter_Widget::get_instance();
